/*******************************************************************************
 *
 *  hal_rfspi.c
 *
 *  Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************/

#include "msp430.h"
#include "hal_rfspi.h"

/***************************************************************************//**
 * @file       hal_rfspi.c
 * @addtogroup hal_rfspi
 * @{
 ******************************************************************************/

/**********************************************************************//**
* @brief  Initializes the ports and serial communication peripherals to
*         communicate with the radio.
*
* @param  none
*
* @return none
**************************************************************************/

void halRfSpiInit(void)
{
	/* Setup CS to be used as GPIO */
	RF_PORT_SEL &= ~RF_SPI_CS;
	RF_PORT_OUT |= RF_SPI_CS;
    RF_PORT_DIR |= RF_SPI_CS;
	
    // Set up pins used by peripheral unit
    RF_PORT_SEL |= RF_SPI_SIMO + RF_SPI_SOMI + RF_SPI_CLK;
    RF_PORT_DIR |= RF_SPI_CLK + RF_SPI_SIMO;
	RF_PORT_DIR &= ~RF_SPI_SOMI;
	
    RF_PORT_REN |= RF_SPI_SOMI;                                   // Pull-Ups on RF Interface SOMI
    RF_PORT_OUT |= RF_SPI_SOMI;                                   
    RF_PORT_SEL |= RF_SPI_SIMO + RF_SPI_SOMI + RF_SPI_CLK;

    // Initialize USCI_B1 for SPI Master operation
    UCB0CTL1 = UCSWRST;                                   // Put state machine in reset
    UCB0CTL0 |= /*UCCKPH +*/ UCMSB + UCMST  + UCMODE_0 + UCSYNC ; // 3-pin, 8-bit SPI master
    // Clock polarity select - The inactive state is high
    // MSB first
    UCB0CTL1 |= /*UCSWRST +*/ UCSSEL_2;                         // Use SMCLK, keep RESET
	
    UCB0BR0 = 0x10;                                          // Initial SPI clock must be <400kHz
    UCB0BR1 = 0;                                           // f_UCxCLK = 16MHz/16 = 1MHz
    // Release for operation
   UCB0IE &= ~(UCTXIE + UCRXIE); 
	   
	UCB0CTL1 &= ~UCSWRST;
    UCB0IFG &= ~UCRXIFG;
}

/**********************************************************************//**
* @brief  Disables the serial communications peripheral and clears the GPIO
*         settings used to communicate with the radio.
*
* @param  none
*
* @return none
**************************************************************************/

void halRfSpiShutDown(void)
{
    UCB0CTL1 |= UCSWRST;
    RF_PORT_SEL &= ~(RF_SPI_SIMO + RF_SPI_SOMI + RF_SPI_CLK);
    RF_PORT_OUT &= ~RF_SPI_CS;
}

/**********************************************************************//**
* @brief  Set the chip select pin high.
*
* @param  none
*
* @return none
**************************************************************************/

void halRfSpiCSHigh(void)
{
    RF_PORT_OUT |= RF_SPI_CS;               // Set CSn high
}

/**********************************************************************//**
* @brief  Set the chip select pin high.
*
* @param  none
*
* @return none
**************************************************************************/

void halRfSpiCSLow(void)
{
    RF_PORT_OUT &= ~RF_SPI_CS;               // Set CSn low
}

/***************************************************************************//**
 * @brief   Read a frame of bytes via SPI
 * @param   pBuffer Place to store the received bytes
 * @param   size Indicator of how many bytes to receive
 * @return  None
 ******************************************************************************/
void halRFSpiReadWriteFrame(uint8_t *pTxBuffer, uint8_t *pRxBuffer, uint16_t size)
{
	
    // Clock the actual data transfer and receive the bytes
    while (size--){
        while (!(UCB0IFG & UCTXIFG)) ;                     // Wait while not ready for TX
        UCB0TXBUF = *pTxBuffer++;                           // Write TX byte
        while (!(UCB0IFG & UCRXIFG)) ;                     // Wait for RX buffer (full)
        *pRxBuffer++ = UCB0RXBUF;
    }
}
/***************************************************************************//**
 * @}
 ******************************************************************************/

